varying vec3 f_view_pos;
varying float f_alpha;
varying float f_alpha_adapt;

uniform sampler2D texture_color;
uniform sampler2D texture_depth;

uniform mat4 proj;
uniform float underwater_factor;

#ifdef QUALITY_HIGH_NORMAL
varying vec3 f_view_normal;
#else
uniform vec3 view_normal;
#endif

#ifdef QUALITY_HIGH_RAY
const float rayStep = 2.0;
const int maxSteps = 25;
const int maxStepsBinary = 5;
#else
const float rayStep = 5.0;
const int maxSteps = 10;
const int maxStepsBinary = 6;
#endif

const float rayStepMin = 0.1;
const float rayStepMax = 1.0;

const float maxDeltaDepth = 2.0;
const float maxDeltaDepthInv = 1.0 / maxDeltaDepth;

const float maxDist = 500.0;
const float alphaBase = 0.9;



vec4 getCoordProj(in vec3 coordHit)
{
	vec4 coordProj = proj * vec4(coordHit, 1.0);
	coordProj.xy /= coordProj.w;
	coordProj.xy = coordProj.xy * 0.5 + 0.5;
	return coordProj;
}

float getDepthView(in vec2 coordProj)
{
	return -texture2D(texture_depth, coordProj.xy).r * VIEW_FRUSTUM_LEN;
}

vec3 rayBinarySearch(in vec3 coordHit, in vec3 dir, out float deltaDepth)
{
	float depth;
	for(int i = 0; i < maxStepsBinary; ++i)
	{
		vec4 coordProj = getCoordProj(coordHit);

		depth = getDepthView(coordProj.xy);
		deltaDepth = coordHit.z - depth;

		if(deltaDepth > 0.0)
			coordHit += dir;
		dir *= 0.5;
		coordHit -= dir;
	}

	vec4 coordProj = getCoordProj(coordHit);
	return vec3(coordProj.xy, depth);
}

vec4 rayCast(in vec3 coordHit, in vec3 dir, out float deltaDepth)
{
	dir *= rayStep;
	for(int i = 0; i < maxSteps; ++i)
	{
		coordHit += dir;
		vec4 coordProj = getCoordProj(coordHit);

		float depth = getDepthView(coordProj.xy);
		deltaDepth = coordHit.z - depth;

		if(deltaDepth < 0.0)
			return vec4(rayBinarySearch(coordHit, dir, deltaDepth), 1.0);
	}
	return vec4(0.0, 0.0, 0.0, 0.0);
}

float getEdgeFactor(in vec2 coords, in float fullMultiplier)
{
	const float adaptX = 0.60;
	vec2 deltaCoords = abs(vec2(0.5, 0.5) - coords.xy);
	float distFactor = 1.0 - 2.0*max(adaptX*deltaCoords.x, deltaCoords.y);
	return clamp(distFactor*fullMultiplier, 0.0, 1.0);
}

void main()
{
	vec2 coordScreen = gl_FragCoord.xy * vec2(SCREEN_WIDTH_INV, SCREEN_HEIGHT_INV);

	const float depthEpsilon = 0.1;
	if(f_alpha < 0.1 || f_view_pos.z-depthEpsilon < getDepthView(coordScreen))
	{
		gl_FragColor = vec4(0.0, 0.0, 0.0, 0.0);
		return;
	}

#ifdef QUALITY_HIGH_NORMAL
	vec3 normal = normalize(f_view_normal);
#else
	vec3 normal = view_normal;
#endif
	vec3 reflected = normalize(reflect(normalize(f_view_pos), normal));

	float deltaDepth = 0.0;
	vec4 coords = rayCast(f_view_pos, reflected * clamp(-f_view_pos.z, rayStepMin, rayStepMax), deltaDepth);

	float deltaDepthFactor = 1.0 - clamp(abs(deltaDepth) * maxDeltaDepthInv, 0.0, 1.0);
	float sampleEdgeFactor = getEdgeFactor(coords.xy, 2.0);

	float distFactor = 1.0 - clamp(-f_view_pos.z / maxDist, 0.0, 1.0);

	gl_FragColor = vec4(texture2D(texture_color, coords.xy).rgb,
						alphaBase *
						f_alpha_adapt *
						coords.w *
						underwater_factor *
						deltaDepthFactor *
						sampleEdgeFactor *
						distFactor *
						clamp(-reflected.z, 0.0, 1.0));
}


